/*-*- Mode: C; c-basic-offset: 8 -*-*/

/***
    This file is part of nss-ipv6literal.

    Copyright 2008 Lennart Poettering

    nss-ipv6literal is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation, either version 2.1
    of the License, or (at your option) any later version.

    nss-ipv6literal is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with nss-ipv6literal. If not, If not, see
    <http://www.gnu.org/licenses/>.
***/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdbool.h>
#include <stdlib.h>
#include <limits.h>
#include <nss.h>
#include <sys/types.h>
#include <netdb.h>
#include <errno.h>
#include <string.h>
#include <stdio.h>
#include <assert.h>
#include <unistd.h>
#include <net/if.h>

#define SUFFIX "ipv6-literal.net"

#define ALIGN(a) (((a+sizeof(void*)-1)/sizeof(void*))*sizeof(void*))

static int check_name(const char *name, char **ipv6str)
{
	char *ret, *p;
	bool zoneidx = false;

	p = strchr(name, '.');

	/* Check if this is an ipv6-literal name */
	if (!p || !*(++p) || (strcasecmp(p, SUFFIX) != 0)) {
                return ENOENT;
	}

	/* copy and terminate address part */
	ret = strndup(name, p-name-1);
	if (!ret) {
                return ENOMEM;
        }

	/* convert iand validate syntax */
	p = ret;
	while (*p) {
		switch (*p) {
		case '-':
			*p = ':';
			break;
		case 's':
			*p = '%';
			zoneidx = true;
			break;
		default:
			if (*p >= '0' && *p <= '9') break;
			if (!zoneidx &&
			    ((*p >= 'a' && *p <= 'f') ||
			     (*p >= 'A' && *p <= 'F'))) break;

			/* no other values are valid */
			free(ret);
			return ENOENT;
		}
		p++;
	}

	*ipv6str = ret;
	return 0;
}

static int get_addr(char *ipv6str, uint8_t *addr, uint32_t *scopeid)
{
	struct addrinfo hints;
	struct sockaddr_in6 *addr6;
	struct addrinfo *res;
	int ret;

	/* set hints */
	memset(&hints, 0, sizeof(struct addrinfo));
	hints.ai_family = AF_INET6;
	hints.ai_socktype = SOCK_STREAM;

	ret = getaddrinfo(ipv6str, NULL, &hints, &res);
	if (ret != 0) {
		return ENOENT;
	}

	if (res->ai_family != AF_INET6 || res->ai_addr == NULL) {
		freeaddrinfo(res);
		return ENOENT;
	}

	addr6 = (struct sockaddr_in6 *)(res->ai_addr);

        memcpy(addr, addr6->sin6_addr.s6_addr, 16);
	if (scopeid) {
	        *scopeid = addr6->sin6_scope_id;
	}

	freeaddrinfo(res);
	return 0;
}

enum nss_status _nss_ipv6literal_gethostbyname4_r(
                const char *name,
                struct gaih_addrtuple **pat,
                char *buffer, size_t buflen,
                int *errnop, int *h_errnop,
                int32_t *ttlp)
{
        struct gaih_addrtuple *r_tuple1;
        size_t l, idx, ms;
        char *ipv6addr;
	int ret;

        l = strlen(name);
        ms = ALIGN(l+1)+ALIGN(sizeof(struct gaih_addrtuple));
        if (buflen < ms) {
                *errnop = ENOMEM;
                *h_errnop = NO_RECOVERY;
                return NSS_STATUS_TRYAGAIN;
        }

	ret = check_name(name, &ipv6addr);
	if (ret) {
		*errnop = ret;
                *h_errnop = HOST_NOT_FOUND;
                return NSS_STATUS_NOTFOUND;
	}

	/* set buffer */
        r_tuple1 = (struct gaih_addrtuple*)buffer;
        idx = ALIGN(sizeof(struct gaih_addrtuple));

        /* fill in IPv6 addr */
	ret = get_addr(ipv6addr, (uint8_t *)r_tuple1->addr, &r_tuple1->scopeid);
	if (ret) {
		free(ipv6addr);
		*errnop = ret;
                *h_errnop = HOST_NOT_FOUND;
                return NSS_STATUS_NOTFOUND;
	}

        /* Copy in hostname */
        memcpy(r_tuple1->name, name, l+1);
        idx += ALIGN(l+1);

        r_tuple1->family = AF_INET6;
        r_tuple1->next = NULL;

        /* Verify the size matches */
        assert(idx == ms);

        *pat = r_tuple1;

        if (ttlp)
                *ttlp = 0;

	free(ipv6addr);
        return NSS_STATUS_SUCCESS;
}

static enum nss_status fill_in_hostent(
                const char *name,
		int af,
                struct hostent *result,
                char *buffer, size_t buflen,
                int *errnop, int *h_errnop,
                int32_t *ttlp,
                char **canonp)
{
        size_t l, idx, ms;
        char *r_addr, *r_name, *r_aliases, *r_addr_list;
	char *ipv6addr;
        size_t alen;
	int ret;

        if (af == AF_UNSPEC)
                af = AF_INET6;

	if (af != AF_INET6) {
		*errnop = ENOENT;
                *h_errnop = HOST_NOT_FOUND;
                return NSS_STATUS_NOTFOUND;
	}

        alen = 16;

        l = strlen(name);
        ms = ALIGN(l+1)+sizeof(char*)+ALIGN(alen)+sizeof(char*)*2;
        if (buflen < ms) {
                *errnop = ENOMEM;
                *h_errnop = NO_RECOVERY;
                return NSS_STATUS_TRYAGAIN;
        }

	ret = check_name(name, &ipv6addr);
	if (ret) {
		*errnop = ret;
                *h_errnop = HOST_NOT_FOUND;
                return NSS_STATUS_NOTFOUND;
	}

        /* First, get address */
        r_addr = buffer;
	ret = get_addr(ipv6addr, (uint8_t *)r_addr, NULL);
	if (ret) {
		free(ipv6addr);
		*errnop = ret;
                *h_errnop = HOST_NOT_FOUND;
                return NSS_STATUS_NOTFOUND;
	}
        idx = ALIGN(alen);

        /* Second, fill in hostname */
        r_name = buffer + idx;
        memcpy(r_name, name, l+1);
        idx += ALIGN(l+1);

        /* Third, create aliases array */
        r_aliases = buffer + idx;
        *(char**) r_aliases = NULL;
        idx += sizeof(char*);

        /* Fourth, add address pointer array */
        r_addr_list = buffer + idx;
        ((char**) r_addr_list)[0] = r_addr;
        ((char**) r_addr_list)[1] = NULL;
        idx += sizeof(char*)*2;

        /* Verify the size matches */
        assert(idx == ms);

        result->h_name = r_name;
        result->h_aliases = (char**) r_aliases;
        result->h_addrtype = af;
        result->h_length = alen;
        result->h_addr_list = (char**) r_addr_list;

        if (ttlp)
                *ttlp = 0;

        if (canonp)
                *canonp = r_name;

	free(ipv6addr);
        return NSS_STATUS_SUCCESS;
}

enum nss_status _nss_ipv6literal_gethostbyname3_r(
                const char *name,
                int af,
                struct hostent *host,
                char *buffer, size_t buflen,
                int *errnop, int *h_errnop,
                int32_t *ttlp,
                char **canonp) {

        return fill_in_hostent(name, af, host, buffer, buflen, errnop, h_errnop, ttlp, canonp);
}

enum nss_status _nss_ipv6literal_gethostbyname2_r(
                const char *name,
                int af,
                struct hostent *host,
                char *buffer, size_t buflen,
                int *errnop, int *h_errnop) {

        return fill_in_hostent(name, af, host, buffer, buflen, errnop, h_errnop, NULL, NULL);
}

enum nss_status _nss_ipv6literal_gethostbyname_r (
                const char *name,
                struct hostent *host,
                char *buffer, size_t buflen,
                int *errnop, int *h_errnop) {

        return fill_in_hostent(name, AF_INET6, host, buffer, buflen, errnop, h_errnop, NULL, NULL);
}
